﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.IO;

namespace Fiskalizacija.DOS.SysTray
{
    public partial class IzvjestajForm : Form
    {
        private BackgroundWorker worker;
        private Configuration config;

        public IzvjestajForm(Configuration config)
        {
            InitializeComponent();
            this.Icon = new Icon(Properties.Resources.MainIcon, new Size(128, 128));
            dtOd.Value = DateTime.Now;
            dtDo.Value = DateTime.Now;
            this.config = config;
        }

        private void btnOk_Click(object sender, EventArgs e)
        {
            if (dtOd.Value.Year != dtDo.Value.Year)
                MessageBox.Show("Datum Od i datum Do moraju biti u istoj godini.\nAko trebate više godina, napravite više izvještaja - jedan izvještaj po godini.", "Upozorenje", MessageBoxButtons.OK, MessageBoxIcon.Information);
            else
            {
                dtOd.Enabled = false;
                dtDo.Enabled = false;
                rbExcel.Enabled = false;
                rbNotepad.Enabled = false;
                btnOk.Visible = false;
                progressBar.Visible = true;

                // pokreni novi background worker
                worker = new BackgroundWorker();
                worker.WorkerSupportsCancellation = true;
                worker.WorkerReportsProgress = true;
                worker.DoWork += new DoWorkEventHandler(worker_DoWork);
                worker.RunWorkerCompleted += new RunWorkerCompletedEventHandler(worker_Completed);
                worker.ProgressChanged += new ProgressChangedEventHandler(worker_ProgressChanged);
                worker.RunWorkerAsync();
            }
        }

        private void worker_DoWork(object sender, DoWorkEventArgs e)
        {
            if (!Directory.Exists(config.LogFolder))
            {
                throw new Exception("Nema datoteka logova (" + config.LogFolder + ") !");
            }

            // pročitaj listu datuma iz foldera LOG - jedan dan = jedan subfolder
            string[] daysFolders = LogFile.GetFoldersForYear(config.LogFolder, dtOd.Value);
            Dictionary<string, string> daysPath = new Dictionary<string, string>();
            foreach (string day in daysFolders)
                daysPath[Path.GetFileName(day)] = day;  // mapiraj folder datuma na apsolutni path

            IzvjestajPoDanima izvjestaj = new IzvjestajPoDanima();

            try
            {
                DateTime dt = dtOd.Value;
                int totDana = (dtDo.Value - dtOd.Value).Days;
                if (totDana <= 0) totDana = 1;

                while (true)    // glavna petlja za čitanje raspona datuma od-do i pripadajućih računa
                {
                    if (dt > dtDo.Value)
                        break;      // prekid ako smo došli do krajnjeg datuma

                    if ((worker.CancellationPending == true))
                    {
                        e.Cancel = true;
                        return;      // prekid ako je pritisnut Cancel
                    }

                    worker.ReportProgress((((dt - dtOd.Value).Days) * 100) / totDana);

                    string dayFolder = dt.ToString("yyMMdd");
                    if (daysPath.ContainsKey(dayFolder))
                    {
                        string[] racuniFiles = Directory.GetFiles(daysPath[dayFolder], "*i*.XML");      // uzmi sve račune u tom danu - samo njihove input xml fileove
                        foreach (string racunFile in racuniFiles)
                        {
                            izvjestaj.Sumiraj(racunFile);
                        }
                    }
                    dt = dt.AddDays(1);              // inkrementiraj dan u godini dok ne dođe preko dtDo
                }


                // piši u csv file
                string outFile = Path.Combine(Path.GetTempPath(), "Fiska" + DateTime.Now.ToString("yyMMddhhmmssff") + (rbExcel.Checked == true ? ".CSV" : ".TXT"));
                using (StreamWriter output = new StreamWriter(outFile))
                {
                    output.WriteLine("Datum     ;  Prvi racun; Zadnji racun; Brojeva racuna; Naknadnih dostava; Razlika brojeva racuna*;     Iznos kn;Iznos bez nak.dost.;;;;*Razlika brojeva racuna = BrojevaRacuna - (ZadnjiRacun - PrviRacun + 1)");
                    IzvjestajPoDanima.Dan total = new IzvjestajPoDanima.Dan();
                    int totalRazlika = 0;
                    foreach (var dan in izvjestaj.Dani)
                    {
                        int razlika = dan.Value.BrojevaRacuna - (dan.Value.ZadnjiRacun - dan.Value.PrviRacun + 1);
                        output.WriteLine(string.Format("{0};{1,12};{2,13};{3,15};{4,18};{5,24};{6,13:0.00};{7,13:0.00};", dan.Key.ToString("dd.MM.yyyy"), dan.Value.PrviRacun, dan.Value.ZadnjiRacun, dan.Value.BrojevaRacuna, dan.Value.NaknadnihDostava, razlika, dan.Value.IznosUkupno / 100.0, dan.Value.IznosBezNaknadnih / 100.0));
                        total.BrojevaRacuna += dan.Value.BrojevaRacuna;
                        total.IznosUkupno += dan.Value.IznosUkupno;
                        total.IznosBezNaknadnih += dan.Value.IznosBezNaknadnih;
                        total.NaknadnihDostava += dan.Value.NaknadnihDostava;
                        totalRazlika += razlika;
                    }
                    output.WriteLine(string.Format("UKUPNO    ;            ;             ;{0,15};{1,18};{2,24};{3,13:0.00};{4,13:0.00};", total.BrojevaRacuna, total.NaknadnihDostava, totalRazlika, total.IznosUkupno / 100.0, total.IznosBezNaknadnih / 100.0));
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);


                    output.WriteLine("Stopa PDV;     Osnovica;    Iznos PDV;");
                    foreach (var pdv in izvjestaj.Pdv)
                    {
                        output.WriteLine(string.Format("{0,8:0.00}%;{1,13:0.00};{2,13:0.00};", pdv.Key / 100.0, pdv.Value.Osnovica / 100.0, pdv.Value.Iznos / 100.0));
                    }
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);


                    output.WriteLine("Operater   ;  Prvi racun; Zadnji racun; Brojeva racuna; Naknadnih dostava;     Iznos kn;    Storno kn;");
                    foreach (var rad in izvjestaj.Radnici)
                    {
                        output.WriteLine(string.Format("{0};{1,12};{2,13};{3,15};{4,18};{5,13:0.00};{6,13:0.00};", rad.Key, rad.Value.PrviRacun, rad.Value.ZadnjiRacun, rad.Value.BrojevaRacuna, rad.Value.NaknadnihDostava, rad.Value.IznosUkupno / 100.0, rad.Value.StornoIznos / 100.0));
                    }
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);
                    output.WriteLine(string.Empty);


                    output.WriteLine("Nacin placanja;     Iznos kn;");
                    foreach (var np in izvjestaj.NacinPlacanja)
                    {
                        output.WriteLine(string.Format("{0,-14};{1,13:0.00};", np.Key, np.Value / 100.0));
                    }
                    

                    // sumnjivi brojevi računa
                    bool header = false;
                    foreach (var allrns in izvjestaj.SumnjivBrRacuna)
                    {
                        if (allrns.Value.Count > 1)
                        {
                            if (header == false)        // isprintaj header samo ako treba
                            {
                                header = true;
                                output.WriteLine(string.Empty);
                                output.WriteLine(string.Empty);
                                output.WriteLine(string.Empty);
                                output.WriteLine(string.Empty);
                                output.WriteLine(string.Empty);
                                output.WriteLine(string.Empty);
                                output.WriteLine("Dupli broj racuna;Datum/Vrijeme      ;     Iznos kn; Naknadna dostava; Zki                            ; File;");
                            }
                            foreach (var dupli in allrns.Value)
                            {
                                string brRacuna = (allrns.Key == dupli.BrojRacuna || dupli.BrojRacuna == 0) ? allrns.Key.ToString() : string.Format("{0} ({1})", allrns.Key, dupli.BrojRacuna);     // moguće je da se razlikuju broj računa u imenu filea i broj računa u XMLu
                                output.WriteLine(string.Format("{0,17};{1,19};{2,13:0.00}; {3,-16};{4};file:///{5};", brRacuna, dupli.DatumVrijeme, dupli.IznosRacuna / 100.0, dupli.NaknadnaDostava, dupli.Zki, dupli.AbsPath));
                            }
                        }
                    }


                    // rupe u brojevima računa
                    List<int> brRacs = new List<int>(izvjestaj.SumnjivBrRacuna.Keys);
                    brRacs.Sort(
                        delegate(int i1, int i2)
                        {
                            return i1.CompareTo(i2);
                        }
                    );
                    header = false;
                    int zadnjiBroj = -1;
                    foreach (int brRac in brRacs)
                    {
                        if (zadnjiBroj > -1)
                        {
                            if (brRac - 1 > zadnjiBroj)
                            {
                                if (header == false)        // isprintaj header samo ako treba
                                {
                                    header = true;
                                    output.WriteLine(string.Empty);
                                    output.WriteLine(string.Empty);
                                    output.WriteLine(string.Empty);
                                    output.WriteLine(string.Empty);
                                    output.WriteLine(string.Empty);
                                    output.WriteLine(string.Empty);
                                    output.WriteLine("Rupe u brojevima racuna");
                                }
                                output.WriteLine(string.Format("{0} do {1}", zadnjiBroj + 1, brRac - 1));
                            }
                        }
                        zadnjiBroj = brRac;
                    }


                }
                System.Diagnostics.Process.Start(outFile);  // otvori outFile u Excelu ili Notepadu
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        private void worker_ProgressChanged(object sender, ProgressChangedEventArgs e)
        {
            this.progressBar.Value = e.ProgressPercentage;
        }

        private void worker_Completed(object sender, RunWorkerCompletedEventArgs e)
        {
            worker.Dispose();
            worker = null;

            dtOd.Enabled = true;
            dtDo.Enabled = true;
            rbExcel.Enabled = true;
            rbNotepad.Enabled = true;
            btnOk.Visible = true;
            progressBar.Visible = false;

            if (e.Error != null)
            {
                MessageBox.Show(e.Error.Message);
            }
        }

        private void IzvjestajForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (worker != null)
                worker.CancelAsync();
        }

    }
}
