﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using ZXing;

namespace Fiskalizacija.DOS.SysTray
{
    public class QR
    {
        private SysTrayForm sysTrayForm;
        private string errorMessage;

        public QR(SysTrayForm sysTrayForm)
        {
            this.sysTrayForm = sysTrayForm;
        }

        public bool WriteQR(string qrText, Configuration config, DateTime date, string brRacuna)
        {
            string destFile = GetOutputPath(config.ResponseFile, config.QRFile, date, brRacuna);
            return WriteQR(qrText, destFile, (QRItemType)config.QRType, config.QRSize, config.QRFactor, config.QRHorPomak);
        }

        public bool WriteQR(string qrText, string destFile, QRItemType qrDestType, Decimal qrSize, Decimal qrFactor, int horPomak)
        {
            errorMessage = null;
            if (qrDestType == QRItemType.None)
            {
                return false;
            }

            if (String.IsNullOrEmpty(qrText))
            {
                deleteFile(destFile);
                return false;
            }
            else
            {
                try
                {
                    System.Drawing.Imaging.ImageFormat imageFormat = System.Drawing.Imaging.ImageFormat.Wmf;
                    switch (qrDestType)
                    {
                        case QRItemType.Nativni:
                            writeNativniQR(qrText, destFile, (int)qrSize, horPomak);
                            break;
                        case QRItemType.NativniStar:
                            writeNativniStar(qrText, destFile, (int)qrSize, horPomak);
                            break;
                        case QRItemType.PNG:
                            imageFormat = System.Drawing.Imaging.ImageFormat.Png;
                            break;
                        case QRItemType.JPG:
                            imageFormat = System.Drawing.Imaging.ImageFormat.Jpeg;
                            break;
                        case QRItemType.GIF:
                            imageFormat = System.Drawing.Imaging.ImageFormat.Gif;
                            break;
                        case QRItemType.TIF:
                            imageFormat = System.Drawing.Imaging.ImageFormat.Tiff;
                            break;
                        case QRItemType.BMP:
                        default:
                            imageFormat = System.Drawing.Imaging.ImageFormat.Bmp;
                            break;
                    }
                    if (imageFormat != System.Drawing.Imaging.ImageFormat.Wmf)
                    {
                        BarcodeWriter bc = new BarcodeWriter();
                        bc.Format = BarcodeFormat.QR_CODE;
                        bc.Options.Hints.Add(EncodeHintType.ERROR_CORRECTION, ZXing.QrCode.Internal.ErrorCorrectionLevel.M);
                        bc.Options.Width = (int)qrSize;
                        bc.Options.Height = (int)qrSize;

                        bc.Options.Margin = 1;
                        bc.Options.PureBarcode = true;
                        using (System.Drawing.Bitmap bitmap = bc.Write(qrText))
                        {
                            if (qrDestType == QRItemType.GSv)
                            {
                                writeQRGSv(bitmap, destFile, horPomak);
                            }
                            else if (qrDestType == QRItemType.ESCAsterisk)
                            {
                                writeQRESCAsterisk(bitmap, destFile, (double)qrFactor, horPomak);
                            }
                            //else if (qrDestType == QRItemType.ESCAsteriskSire)
                            //{
                            //    writeQRESCAsterisk(bitmap, destFile, true, horPomak);
                            //}
                            else if (qrDestType == QRItemType.ESCK)
                            {
                                writeQRESCK(bitmap, destFile, (double)qrFactor, horPomak);
                            }
                            //else if (qrDestType == QRItemType.ESCKSire)
                            //{
                            //    writeQRESCK(bitmap, destFile, true, horPomak);
                            //}
                            else
                            {
                                bitmap.Save(destFile, imageFormat);
                            }
                        }
                    }
                }
                catch (Exception ex)
                {
                    errorMessage = "Greška kreiranja QR: " + ex.Message;
                    LogError(errorMessage);
                    deleteFile(destFile);
                    return false;
                }
            }
            return true;
        }

        private unsafe void writeQRGSv(System.Drawing.Bitmap bitmap, string filename, int horPomak)
        {
            List<byte> bytes = new List<byte>();
            for (int i=0; i<horPomak; i++)
            {
                bytes.Add(32);
            }
            byte m = 0;

            System.Drawing.Imaging.BitmapData bData = bitmap.LockBits(new System.Drawing.Rectangle(0, 0, bitmap.Width, bitmap.Height), System.Drawing.Imaging.ImageLockMode.ReadOnly, bitmap.PixelFormat);

            /*This time we convert the IntPtr to a ptr*/
            byte* scan0 = (byte*)bData.Scan0.ToPointer();

            int bitonalWidth = (bitmap.Width + 7) / 8;
            byte xL = (byte)(bitonalWidth);
            byte xH = (byte)(bitonalWidth / 256);
            byte yL = (byte)(bitmap.Height);
            byte yH = (byte)(bitmap.Height / 256);

            // GS v 0
            bytes.Add(29);      // GS
            bytes.Add(118);     // v
            bytes.Add(48);      // 0
            bytes.Add(m);
            bytes.Add(xL);
            bytes.Add(xH);
            bytes.Add(yL);
            bytes.Add(yH);
            for (int y = 0; y < bData.Height; y++)
            {
                byte* data = scan0 + y * bData.Stride;
                int b = 0;
                int b1;
                int mask = 128;
                for (int x = 0; x < bData.Width; x++)
                {
                    b1 = *data;
                    b1 = b1 & mask;
                    b |= b1;
                    mask = mask >> 1;
                    if (mask == 0)
                    {
                        bytes.Add((byte)(~b));
                        b = 0;
                        mask = 128;
                    }
                    data += 3;
                }
                if (mask != 128)
                {
                    bytes.Add((byte)(~b));
                }
            }
            bitmap.UnlockBits(bData);

            System.IO.File.WriteAllBytes(filename, bytes.ToArray());
        }

        private unsafe void writeQRESCAsterisk(System.Drawing.Bitmap bitmap, string filename, double ratioFactor, int horPomak)
        {
            if (ratioFactor < 0.95 || ratioFactor > 1.05)
            {
                System.Drawing.Size size = new System.Drawing.Size((int)Math.Round(bitmap.Width * ratioFactor), bitmap.Height);
                using (System.Drawing.Bitmap bitmap2 = (new System.Drawing.Bitmap(bitmap, size)))
                {
                    writeQRESCAsterisk(bitmap2, filename, horPomak);
                }
            }
            else
            {
                writeQRESCAsterisk(bitmap, filename, horPomak);
            }
        }

        private unsafe void writeQRESCAsterisk(System.Drawing.Bitmap bitmap, string filename, int horPomak)
        {
            List<byte> bytes = new List<byte>();
            byte m = 0;

            System.Drawing.Imaging.BitmapData bData = bitmap.LockBits(new System.Drawing.Rectangle(0, 0, bitmap.Width, bitmap.Height), System.Drawing.Imaging.ImageLockMode.ReadOnly, bitmap.PixelFormat);

            int bpp = 3;
            if (bData.PixelFormat == System.Drawing.Imaging.PixelFormat.Format32bppArgb || bData.PixelFormat == System.Drawing.Imaging.PixelFormat.Format32bppPArgb || bData.PixelFormat == System.Drawing.Imaging.PixelFormat.Format32bppRgb)
            {
                bpp = 4;
            }

            /*This time we convert the IntPtr to a ptr*/
            byte* scan0 = (byte*)bData.Scan0.ToPointer();

            byte nh = (byte)(bitmap.Width / 256);
            byte nl = (byte)(bitmap.Width % 256);
            for (int y = 0; y < bData.Height; y += 8)
            {
                for (int i = 0; i < horPomak; i++)
                {
                    bytes.Add(32);
                }
                bytes.Add(27);              // ESC
                bytes.Add(42);              // *
                bytes.Add(m);
                bytes.Add(nl);
                bytes.Add(nh);
                int endStrip = Math.Min(8, bData.Height - y);

                byte* data = scan0 + y * bData.Stride;
                for (int x = 0; x < bData.Width; x++)
                {
                    int b = 0;
                    int b1;
                    int mask = 128;
                    for (int i = 0; i < endStrip; i++)
                    {
                        if (data[i * bData.Stride] > 127)
                            b1 = 0;
                        else
                            b1 = 255;
                        b1 = b1 & mask;
                        b |= b1;
                        mask = mask >> 1;
                    }
                    bytes.Add((byte)b);
                    data += bpp;
                }
                bytes.Add(27);  // ESC
                bytes.Add(51);  // 3
                bytes.Add(16);
                bytes.Add(13);
                bytes.Add(10);
            }
            bytes.Add(27);  // ESC
            bytes.Add(50);  // 2
            bitmap.UnlockBits(bData);

            System.IO.File.WriteAllBytes(filename, bytes.ToArray());
        }

        private unsafe void writeQRESCK(System.Drawing.Bitmap bitmap, string filename, double ratioFactor, int horPomak)
        {
            if (ratioFactor < 0.95 || ratioFactor > 1.05)
            {
                System.Drawing.Size size = new System.Drawing.Size((int)Math.Round(bitmap.Width * ratioFactor), bitmap.Height);
                using (System.Drawing.Bitmap bitmap2 = (new System.Drawing.Bitmap(bitmap, size)))
                {
                    writeQRESCK(bitmap2, filename, horPomak);
                }
            }
            else
            {
                writeQRESCK(bitmap, filename, horPomak);
            }
        }

        private unsafe void writeQRESCK(System.Drawing.Bitmap bitmap, string filename, int horPomak)
        {
            List<byte> bytes = new List<byte>();
            
            System.Drawing.Imaging.BitmapData bData = bitmap.LockBits(new System.Drawing.Rectangle(0, 0, bitmap.Width, bitmap.Height), System.Drawing.Imaging.ImageLockMode.ReadOnly, bitmap.PixelFormat);

            /*This time we convert the IntPtr to a ptr*/
            byte* scan0 = (byte*)bData.Scan0.ToPointer();

            int bpp = 3;
            if (bData.PixelFormat == System.Drawing.Imaging.PixelFormat.Format32bppArgb || bData.PixelFormat == System.Drawing.Imaging.PixelFormat.Format32bppPArgb || bData.PixelFormat == System.Drawing.Imaging.PixelFormat.Format32bppRgb)
            {
                bpp = 4;
            }

            byte nh = (byte)(bitmap.Width / 256);
            byte nl = (byte)(bitmap.Width % 256);
            for (int y = 0; y < bData.Height; y += 8)
            {
                for (int i = 0; i < horPomak; i++)
                {
                    bytes.Add(32);
                }

                bytes.Add(27);              // ESC
                bytes.Add(75);              // K
                bytes.Add(nl);
                bytes.Add(nh);
                int endStrip = Math.Min(8, bData.Height - y);

                byte* data = scan0 + y * bData.Stride;
                for (int x = 0; x < bData.Width; x++)
                {
                    int b = 0;
                    int b1;
                    int mask = 128;
                    for (int i = 0; i < endStrip; i++)
                    {
                        if (data[i * bData.Stride] > 127)
                            b1 = 0;
                        else
                            b1 = 255;
                        b1 = b1 & mask;
                        b |= b1;
                        mask = mask >> 1;
                    }
                    bytes.Add((byte)b);
                    data += bpp;
                }
                bytes.Add(27);  // ESC
                bytes.Add(48);  // 0        (Set line spacing to 3 mm)
                bytes.Add(13);
                bytes.Add(10);
            }
            bytes.Add(27);  // ESC
            bytes.Add(122); // z
            bytes.Add(1);   //              (Set line spacing to 4 mm)
            bitmap.UnlockBits(bData);

            System.IO.File.WriteAllBytes(filename, bytes.ToArray());
        }

        private unsafe void writeNativniQR(string text, string filename, int qrSize, int horPomak)
        {
            byte[] bytes = Encoding.ASCII.GetBytes(text);
            int store_len = bytes.Length + 3;
            byte store_pL = (byte)(store_len % 256);
            byte store_pH = (byte)(store_len / 256);
            byte size = (byte)(qrSize / 25);
            if (size < 1) size = 1;
            if (size > 15) size = 15;
            byte[] start = new byte[]
            {
                //27, 64,
                29, 40, 107, 4, 0, 49, 65, 50, 0,                     // 165 This command sets the QR Code mode 2
                29, 40, 107, 3, 0, 49, 67, size,                      // 167 This command sets the size of the QR Code module to n dots. (1 ≤ size ≤ 16)
                29, 40, 107, 3, 0, 49, 69, 48,                        // 169 This command sets the error correction level for QR Code. n=48
                29, 40, 107, store_pL, store_pH, 49, 80, 48,          // 180 This command saves symbol data of the QR Code to the symbol storage area.
                //50,51,52,53,54,55,56,111,112,113,114,115,116,117,
                //29, 40, 107, 3, 0, 49, 81, 48,                      // 181 This command encodes and prints QR Code symbol data saved in the symbol storage area
            };
            byte[] end = new byte[]
            {
                29, 40, 107, 3, 0, 49, 81, 48,                      // 181 This command encodes and prints QR Code symbol data saved in the symbol storage area
            };

            byte[] startAlignment, endAlignment;
            if (horPomak == 0)
            {
                startAlignment = endAlignment = new byte[0];
            }
            else if (horPomak <= 15)
            {
                startAlignment = new byte[] { 27, 97, 1, };     // center alignment
                endAlignment = new byte[] { 27, 97, 0, };       // left alignment
            }
            else
            {
                startAlignment = new byte[] { 27, 97, 2, };     // right alignment
                endAlignment = new byte[] { 27, 97, 0, };       // left alignment
            }

            using (FileStream fs = new FileStream(filename, FileMode.Create))
            {
                using (BinaryWriter bw = new BinaryWriter(fs))
                {
                    bw.Write(startAlignment);
                    bw.Write(start);
                    bw.Write(bytes);
                    bw.Write(end);
                    bw.Write(endAlignment);
                }
            }
        }

        private unsafe void writeNativniPDF417(string text, string filename, int qrSize, int horPomak)
        {
            byte[] bytes = Encoding.ASCII.GetBytes(text);
            int store_len = bytes.Length + 3;
            byte store_pL = (byte)(store_len % 256);
            byte store_pH = (byte)(store_len / 256);
            byte size = (byte)(qrSize / 40);
            if (size < 2) size = 2;
            if (size > 8) size = 8;
            byte[] start = new byte[]
            {
                //27, 64,
                29, 40, 107, 3, 0, 48, 67, size,                      // 167 This command sets the size of the module to n dots. (2 ≤ size ≤ 8)
                29, 40, 107, 3, 0, 48, 68, size,                      // 167 This command sets the size of the module to n dots. (2 ≤ size ≤ 8)
                29, 40, 107, 4, 0, 48, 69, 48, 50,                    // 169 This command sets the error correction level. n=50, level=2
                29, 40, 107, store_pL, store_pH, 48, 80, 48,          // 180 This command saves symbol data of the QR Code to the symbol storage area.
            };
            byte[] end = new byte[]
            {
                29, 40, 107, 3, 0, 48, 81, 48,                      // 181 This command encodes and prints symbol data saved in the symbol storage area
            };
            
            using (FileStream fs = new FileStream(filename, FileMode.Create))
            {
                using (BinaryWriter bw = new BinaryWriter(fs))
                {
                    bw.Write(start);
                    bw.Write(bytes);
                    for (int i = 0; i < horPomak; i++)
                    {
                        bw.Write((byte)32);       // spacevi
                    }
                    bw.Write(end);
                }
            }
        }

        private unsafe void writeNativniStar(string text, string filename, int qrSize, int horPomak)
        {
            byte[] bytes = Encoding.ASCII.GetBytes(text);
            int store_len = bytes.Length;
            byte store_pL = (byte)(store_len % 256);
            byte store_pH = (byte)(store_len / 256);
            byte size = (byte)(qrSize / 25);
            if (size < 1) size = 1;
            if (size > 15) size = 15;
            byte[] start = new byte[]
            {
                27, 29, (byte)'y', (byte)'S', (byte)'0', 2,     // izbor print kontrol mod-a
                27, 29, (byte)'y', (byte)'S', (byte)'1', 0,    // error correction qr-koda
                27, 29, (byte)'y', (byte)'S', (byte)'2', size,    // velicina qr-koda
                27, 29, (byte)'y', (byte)'D', (byte)'1', 0, store_pL, store_pH,
            };
            byte[] end = new byte[]
            {
                27, 29, (byte)'y', (byte)'I',
                27, 29, (byte)'y', (byte)'P',
            };

            using (FileStream fs = new FileStream(filename, FileMode.Create))
            {
                using (BinaryWriter bw = new BinaryWriter(fs))
                {
                    bw.Write(start);
                    bw.Write(bytes);
                    bw.Write(end);
                }
            }
        }

        public static string GetOutputPath(string responsePath, string qrPath, DateTime datum, string brRacuna)
        {
            if (qrPath.StartsWith("."))
            {
                qrPath = System.IO.Path.Combine(System.IO.Path.GetDirectoryName(responsePath), qrPath);
            }

            qrPath = qrPath.Replace("%GOD%", datum.Year.ToString());
            qrPath = qrPath.Replace("%DATUM%", datum.ToString("yyMMdd"));
            qrPath = qrPath.Replace("%RACUN%", brRacuna);

            string folder = System.IO.Path.GetDirectoryName(qrPath);
            if (!System.IO.Directory.Exists(folder))
            {
                System.IO.Directory.CreateDirectory(folder);
            }

            return qrPath;
        }

        public bool HasError()
        {
            return !String.IsNullOrEmpty(errorMessage);
        }

        private void LogError(string message)
        {
            sysTrayForm.ShowBalloon(false, message);
            sysTrayForm.LogForm.Add(false, message);
        }

        private void deleteFile(string filename)
        {
            try { File.Delete(filename); }
            catch (Exception) { }
        }

    }
}
