﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Windows.Forms;
using System.ComponentModel;

namespace Fiskalizacija.DOS.SysTray
{
    public class SysTrayForm : Form
    {
        private NotifyIcon trayIcon;
        private ContextMenu trayMenu;
        private Configuration config;
        private readonly string currentPath;
        private RacunWorker racunWorker;
        private LogForm logForm = new LogForm();
        private readonly Icon normalIcon = new Icon(Properties.Resources.MainIcon, new Size(16, 16));
        private readonly Icon errorIcon = new Icon(Properties.Resources.IconError, new Size(16, 16));
        private readonly Icon offlineIcon = new Icon(Properties.Resources.OfflineIcon, new Size(16, 16));
        private MenuItem offlineMenuItem = null;
        private bool isOfflineMode = false;
        private IzvjestajForm izvjestajForm = null;
        private RadnoVrijemeForm radnoVrijemeForm = null;
        private readonly QR qr;

        public SysTrayForm(string currentPath)
        {
            this.currentPath = currentPath;
            qr = new QR(this);
            // Create a tray menu.
            trayMenu = new ContextMenu();
            trayMenu.MenuItems.Add("Namještanja", OnConfiguration);
            trayMenu.MenuItems.Add("Test konekcije (echo)", OnEcho);
            trayMenu.MenuItems.Add("Radno vrijeme", OnRadnoVrijeme);
            MenuItem logMenuItem = trayMenu.MenuItems.Add("Log");
            logMenuItem.MenuItems.Add("Trenutni log", OnLog);
            logMenuItem.MenuItems.Add("Promet po danima", OnPrometPoDanima);
            offlineMenuItem = trayMenu.MenuItems.Add("Offline mode", OnOfflineMode); 
            trayMenu.MenuItems.Add("Izlaz iz programa", OnExit);
                     
            // Create a tray icon.
            trayIcon = new NotifyIcon();
            trayIcon.Text = "Fiskalizacija";
            trayIcon.Icon = normalIcon;

            // Add menu to tray icon and show it.
            trayIcon.ContextMenu = trayMenu;
            trayIcon.Visible = true;
        }

        protected override void OnLoad(EventArgs e)
        {
            Visible = false; // Hide form window.
            ShowInTaskbar = false; // Remove from taskbar.
            config = Configuration.Load(currentPath);  // učitaj namještanja
            offlineMenuItem.Visible = config.Offline;  // sakrij/prikaži menu item Offline mode

            // pokreni glavni worker
            if (!string.IsNullOrEmpty(config.RequestFile))
            {
                racunWorker = new RacunWorker(this, Configuration.Load(currentPath));
                if (racunWorker.startaoOk == false)
                    this.Close();
            }

            base.OnLoad(e);
        }

        private void OnConfiguration(object sender, EventArgs e)
        {
            if (ProvjeriPassword() == DialogResult.OK)
            {
                using (ConfigurationForm configForm = new ConfigurationForm(config, currentPath, qr))
                {
                    configForm.ShowDialog(this);

                    if (configForm.NeedsRefresh && racunWorker != null)
                    {
                        // ubij stari worker
                        racunWorker.Stop();
                        // pokreni novi worker
                        racunWorker = new RacunWorker(this, Configuration.Load(currentPath));

                        // ažuriraj menu Offline mode
                        offlineMenuItem.Visible = config.Offline;
                        if (config.Offline == false) IsOfflineMode = false;
                    }
                }
            }
        }

        private void OnLog(object sender, EventArgs e)
        {
            // prikaži log dialog ili ga stavi u fokus ako je već prikazan
            if (logForm.Visible == false)
                logForm.Show(this);
            else if (logForm.WindowState == FormWindowState.Minimized)
                Invoke(new MethodInvoker(delegate { logForm.WindowState = FormWindowState.Normal; }));
            else
                Invoke(new MethodInvoker(delegate { logForm.Activate(); }));
        }

        private void OnRadnoVrijeme(object sender, EventArgs e)
        {
            if (ProvjeriPassword() == DialogResult.OK)
            {
                if (radnoVrijemeForm == null)
                {
                    radnoVrijemeForm = new RadnoVrijemeForm(this, config);
                    radnoVrijemeForm.ShowDialog(this);
                    radnoVrijemeForm = null;
                }
                else
                {
                    if (radnoVrijemeForm.WindowState == FormWindowState.Minimized)
                        Invoke(new MethodInvoker(delegate { radnoVrijemeForm.WindowState = FormWindowState.Normal; }));
                    else
                        Invoke(new MethodInvoker(delegate { radnoVrijemeForm.Activate(); }));
                }
            }
        }

        private void OnPrometPoDanima(object sender, EventArgs e)
        {
            if (ProvjeriPassword() == DialogResult.OK)
            {
                if (izvjestajForm == null)
                {
                    izvjestajForm = new IzvjestajForm(config);
                    izvjestajForm.ShowDialog(this);
                    izvjestajForm = null;
                }
                else
                {
                    if (izvjestajForm.WindowState == FormWindowState.Minimized)
                        Invoke(new MethodInvoker(delegate { izvjestajForm.WindowState = FormWindowState.Normal; }));
                    else
                        Invoke(new MethodInvoker(delegate { izvjestajForm.Activate(); }));
                }
            }
        }

        private void OnEcho(object sender, EventArgs e)
        {
            Echo.Test(this, config);
        }

        private void OnOfflineMode(object sender, EventArgs e)
        {
            if (isOfflineMode == false)
            {
                offlineMenuItem.Text = "Vrati Online mode";
                IsOfflineMode = true;
                ShowBalloon(true, "Offline mode je uključen");
            }
            else
            {
                offlineMenuItem.Text = "Offline mode";
                IsOfflineMode = false;
            }
        }

        public bool IsOfflineMode
        {
            get { return isOfflineMode; }
            set
            {
                isOfflineMode = value;
                SetIcon(true);
            }
        }

        private void OnExit(object sender, EventArgs e)
        {
            if (ProvjeriPassword() == DialogResult.OK)
            {
                if (racunWorker != null)
                    racunWorker.Stop();
                logForm.Dispose();
                Application.Exit();
            }
        }

        protected override void Dispose(bool isDisposing)
        {
            if (isDisposing)
            {
                // Release the icon resource.
                trayIcon.Dispose();
            }
            base.Dispose(isDisposing);
        }

        public void ShowBalloon(bool isOk, string message)
        {
            trayIcon.Visible = true;
            SetIcon(isOk);
            if (isOk)
            {
                trayIcon.ShowBalloonTip(900, "Fiskalizacija", message, ToolTipIcon.Info);
            }
            else
            {
                trayIcon.ShowBalloonTip(1000, "Greška fiskalizacije", message, ToolTipIcon.Error);
            }
        }

        public void SetIcon(bool isOk)
        {
            if (isOk)
            {
                if (IsOfflineMode)
                {
                    trayIcon.Icon = offlineIcon;
                    trayIcon.Text = "Fiskalizacija (" + config.OznakaPoslProstora + "/" + config.OznakaNaplatnogUredjaja + ") - OFFLINE";
                }
                else
                {
                    trayIcon.Icon = normalIcon;
                    trayIcon.Text = "Fiskalizacija (" + config.OznakaPoslProstora + "/" + config.OznakaNaplatnogUredjaja + ")";
                }
            }
            else
            {
                trayIcon.Icon = errorIcon;
                trayIcon.Text = "Fiskalizacija ne radi (" + config.OznakaPoslProstora + "/" + config.OznakaNaplatnogUredjaja+ ")";
            }
        }

        public LogForm LogForm
        {
            get { return logForm; }
        }

        public QR QR
        {
            get { return qr; }
        }

        private DialogResult ProvjeriPassword()
        {
            DialogResult dlgResult = DialogResult.OK;
            if (!string.IsNullOrEmpty(config.Password))
            {
                using (PasswordForm passwordForm = new PasswordForm(config))
                {
                    dlgResult = passwordForm.ShowDialog(this);
                }
            }
            return dlgResult;
        }
    }
}
